<?php namespace App\Controllers\Api\Ipn;

use App\Controllers\BaseController;
use App\Models\DepositMethodsModel;
use App\Models\PlansModel;
use App\Models\SubscribesModel;
use App\Models\TransactionsModel;
use App\Models\PlansExtraModel;
use App\Models\SubscribesExtraModel;
use App\Models\TransactionsExtraModel;
use App\Models\UsersModel;
use CodeIgniter\HTTP\ResponseInterface;
use CodeIgniter\Database\Exceptions\DatabaseException;

class Paystack extends BaseController
{
    protected $paystackUrl = 'https://api.paystack.co/transaction/verify/';
    protected $method;

    public function __construct()
    {
        // Không gọi parent::__construct() nếu lớp cha không có phương thức khởi tạo
        $depositMethods = new DepositMethodsModel();
        $this->method = $depositMethods->where([
            'name' => 'Paystack',
            'status' => 1
        ])->first();

        if (!$this->method) {
            throw new DatabaseException('Paystack method not found or inactive.');
        }
    }

    public function index(): ResponseInterface
    {
        try {
            $payload = $this->request->getJSON(true);

            if (!$payload) {
                throw new \Exception("Invalid payload.");
            }

            // Lược bỏ xác thực chữ ký Webhook Secret

            $event = $payload;

            if (!isset($event['data'])) {
                throw new \Exception("Invalid event data.");
            }

            $data = $event['data'];
            $eventType = $event['event'];

            // Lấy custom_id từ metadata
            $customId = $data['metadata']['custom_id'] ?? null;
            if ($customId === null) {
                throw new \Exception("Custom ID not found in event data.");
            }

            // Phân biệt giữa thanh toán plan và plan_extra
            if (strpos($customId, 'EXTRA_') === 0) {
                $this->processEventExtra($data, $eventType);
            } else {
                $this->processEvent($data, $eventType);
            }

            return $this->respond(["status" => "ok"], 200);
        } catch (\Exception $e) {
            log_message('error', 'Paystack IPN Error: ' . $e->getMessage());
            return $this->respond(["message" => $e->getMessage()], 400);
        }
    }

    private function processEvent($data, $eventType)
    {
        $db = \Config\Database::connect();
        $db->transStart();

        try {
            if ($eventType === 'charge.success') {
                $this->handleChargeSuccess($data);
            } else {
                // Loại bỏ các sự kiện không cần xử lý
                log_message('info', 'Unhandled Paystack event type: ' . $eventType);
            }

            $db->transComplete();
        } catch (\Exception $e) {
            $db->transRollback();
            throw $e;
        }
    }

    private function processEventExtra($data, $eventType)
    {
        $db = \Config\Database::connect();
        $db->transStart();

        try {
            if ($eventType === 'charge.success') {
                $this->handleChargeExtraSuccess($data);
            } else {
                log_message('info', 'Unhandled Paystack extra event type: ' . $eventType);
            }

            $db->transComplete();
        } catch (\Exception $e) {
            $db->transRollback();
            throw $e;
        }
    }

    private function handleChargeSuccess($data)
    {
        $subscribes = new SubscribesModel();
        $plans = new PlansModel();
        $transactions = new TransactionsModel();
        $users = new UsersModel();

        $customID = $data['metadata']['custom_id'];
        $customValues = explode("_", $customID);

        if (count($customValues) !== 3) {
            throw new \Exception("Invalid custom ID format.");
        }

        list($plan_id, $user_id, $app_id) = $customValues;

        // Lấy thông tin từ bảng plans
        $plan = $plans->where("id", $plan_id)->first();
        if (!$plan) {
            throw new \Exception("Plan not found.");
        }

        $amount_total = $plan["price"];
        $currentDate = date('Y-m-d');
        $newDate = date('Y-m-d', strtotime('+' . $plan['count'] . ' month', strtotime($currentDate)));

        // Kiểm tra nếu đã có subscribe cho app_id và user_id và còn trong thời hạn thì không tạo thêm
        $existing_subscribe = $subscribes->where("user_id", $user_id)
            ->where("app_id", $app_id)
            ->where("expires_at >", strtotime($currentDate))
            ->first();

        if ($existing_subscribe) {
            log_message('error', 'Subscribe already exists for user_id ' . $user_id . ' and app_id ' . $app_id . ' within the period.');
            return; // Dừng lại, không tạo thêm subscribe mới
        }

        // Thêm đăng ký mới vào bảng subscribes
        $subscribes->insert([
            "subscribe_external_id" => $data['reference'],
            "customer_external_id"  => $data['customer']['email'],
            "plan_id"               => $plan_id,
            "user_id"               => $user_id,
            "expires_at"            => strtotime($newDate),
            "app_id"                => $app_id,
            "price"                 => $amount_total,
            "uid"                   => bin2hex(random_bytes(10)),
            "is_active"             => 1,
            "remaining_count"       => $plan["build_count"],
            "method_id"             => $this->method['id']
        ]);

        // Thêm giao dịch vào bảng transactions
        $transactions->insert([
            'uid' => bin2hex(random_bytes(10)),
            'amount' => $amount_total,
            'status' => 1,  // Thành công
            'created_at' => time(),
            'updated_at' => time(),
            'method_id' => $this->method['id'],
            'subscribe_external_id' => $data['reference'],
            'external_uid' => $data['reference']
        ]);
    }

    private function handleChargeExtraSuccess($data)
    {
        $subscribes_extra = new SubscribesExtraModel();
        $plansExtra = new PlansExtraModel();
        $transactions_extra = new TransactionsExtraModel();
        $subscribes = new SubscribesModel();

        $customID = $data['metadata']['custom_id'];
        // Loại bỏ tiền tố 'EXTRA_'
        if (strpos($customID, 'EXTRA_') === 0) {
            $customID = substr($customID, strlen('EXTRA_'));
        } else {
            throw new \Exception("Invalid custom ID format.");
        }
        $customValues = explode("_", $customID);

        if (count($customValues) !== 3) {
            throw new \Exception("Invalid custom ID format.");
        }

        list($plan_extra_id, $user_id, $app_id) = $customValues;

        $plan_extra = $plansExtra->where("id", $plan_extra_id)->first();
        if (!$plan_extra) {
            throw new \Exception("Plan extra not found.");
        }
        $amount_extra_total = $plan_extra["price"];

        // Thêm đăng ký mới vào bảng `subscribes_extra`
        $subscribes_extra->insert([
            "subscribe_extra_external_id" => $data['reference'],
            "customer_extra_external_id"  => $data['customer']['email'],
            "plan_extra_id"               => $plan_extra_id,
            "user_id"                     => $user_id,
            "app_id"                      => $app_id,
            "price"                       => $amount_extra_total,
            "uid"                         => bin2hex(random_bytes(10)),
            "is_active"                   => 1,
            "method_id"                   => $this->method['id'],
            "build_count_extra"           => $plan_extra['build_count_extra']
        ]);

        // Thêm giao dịch vào bảng `transactions_extra`
        $transactions_extra->insert([
            'uid' => bin2hex(random_bytes(10)),
            'amount' => $amount_extra_total,
            'status' => 1,  // Thành công
            'created_at' => time(),
            'updated_at' => time(),
            'method_id' => $this->method['id'],
            'subscribe_extra_external_id' => $data['reference'],
            'external_extra_uid' => $data['reference']
        ]);

        // Cập nhật remaining_count trong bảng subscribes
        $subscribes->set('remaining_count', 'remaining_count + ' . $plan_extra["build_count_extra"], false)
                   ->where("user_id", $user_id)
                   ->where("app_id", $app_id)
                   ->update();
    }

    public function capture(): ResponseInterface
    {
        // Xử lý chuyển hướng sau thanh toán thành công
        $settings = new \App\Libraries\Settings();
        $frontUrl = $settings->get_config("site_url");
        return redirect()->to($frontUrl . 'private/profile/subscribe');
    }

    public function captureExtra(): ResponseInterface
    {
        // Xử lý chuyển hướng sau thanh toán bổ sung thành công
        $settings = new \App\Libraries\Settings();
        $frontUrl = $settings->get_config("site_url");
        return redirect()->to($frontUrl . 'private/profile/subscribe_extra');
    }
}
